C# Language Design Notes Feb 29, 2016
=====================================

Discussion for these notes can be found at https://github.com/dotnet/roslyn/issues/9330.

## Catch up edition (deconstruction and immutable object creation)

Over the past couple of months various design activities took place that weren't documented in design notes. The following is a summary of the state of design regarding positional deconstruction, with-expressions and object initializers for immutable types.

# Philosophy

We agree on the following design tenets:

Positional deconstruction, with-expressions and object initializers are separable features, enabled by the presence of certain API patterns on types that can be expressed manually, as well as generated by other language features such as records.

# API Patterns

API patterns for a language feature facilitate two things:

* Provide actual APIs to call _at runtime_ when the language feature is used
* Inform the compiler _at compile time_ about how to generate code for the feature

It turns out the biggest design challenges are around the second part. Specifically, all these API patterns turn out to need to bridge between positional and name-based expressions of the members of types. How each API pattern does that is a central question of its design.

Assume the following running example:

``` c#
public class Person
{
  public string FirstName { get; }
  public string LastName { get; }

  public Person(string firstName, string lastName)
  {
    FirstName = firstName;
    LastName = lastName;
  }
}
```

In the following we'll consider extending and changing this type to expose various API patterns as we examine the individual language features.

Here's an example of using the three language features:

``` c#
var p = new Person { FirstName = "Mickey", LastName = "Mouse" }; // object initializer
if (p is Person("Mickey", *)) // positional deconstruction
{
  return p with { FirstName = "Minney" }; // with-expression
}
```

Semantically this corresponds to something like this:

``` c#
var p = new Person("Mickey", "Mouse"); // constructor call
if (p.FirstName == "Mickey") // property access
{
  return new Person("Minney", p.LastName); // constructor call
}
```

Notice how the new features that use property names correspond to API calls using positional parameters, whereas the feature that uses positions corresponds to member access by name!

# Object initializers for immutable objects
(See e.g. #229)

This feature allows an object initializer for which assignable properties are not found, to fall back to a constructor call taking the properties' new values as arguments.

``` c#
new Person { FirstName = "Mickey", LastName = "Mouse" }
```
becomes
``` c#
new Person("Mickey", "Mouse")
```

The question then is: how does the compiler decide to pass the given FirstName as the first argument? Somehow it needs clues from the `Person` type as to which properties correspond to which constructor parameters. These clues cannot just be the constructor body: we need this to work across assemblies, so the clues must be evident from metadata.

Here are some options:

1: The type or constructor explicitly includes metadata for this purpose, e.g. in the form of attributes.
2: The names of the constructor parameters must match exactly the names of the corresponding properties.

The former is unattractive because it requires the type's author to write those attributes. It requires the type to be explicitly edited for the purpose.

The latter is better in that it doesn't require extra API elements. However, API design guidelines stipulate that public properties start with uppercase, and parameters start with lower case. This pattern would break that, and for the same reason is highly unlikely to apply to any existing types.

This leads us to:

3: The names of the constructor parameters must match the names of the corresponding properties, _modulo case_!

This would allow a large number of existing types to just work (including the example above), but at the cost of introducing case insensitivity to this part of the C# language. 

# With-expressions
(see e.g. #5172)

With-expressions are similar to object initializers, except that they provide a source object from which to copy all the properties that aren't specified. Thus it seems reasonable to use a similar strategy for compilation; to call a constructor, this time filling in missing properties by accessing those on the source object.

Thus the same strategies as above would apply to establish the connection between properties and constructor parameters.

``` c#
p with { FirstName = "Minney" }
```
becomes
``` c#
new Person("Minney", p.LastName)
```

However, there's a hitch: if the runtime source object is actually of a derived type with more properties than are known from its static type, it would typically be expected that those are copied over too. In that case, the static type is also likely to be abstract (most base types are), so it wouldn't offer a callable constructor.

For this situation there needs to be a way that an abstract base class can offer "with-ability" that correctly copies over members of derived types. The best way we can think of is to offer a virtual `With` method, as follows:

``` c#
public abstract class Person
{
  ...
  public abstract Person With(string firstName, string lastName);
}
``` 

In the presence of such a `With` method we would generate a with expression to call that instead of the constructor:

``` c#
p.With("Minney", p.LastName)
```

We can decide whether to make with-expressions _require_ a `With` method, or fall back to constructor calls in its absence.

If we _require_ a `With` method, that makes for less interoperability with existing types. However, it gives us new opportunities for how to provide the position/name mapping metadata thorugh the declaration of that `With` method: For instance, we could introduce a new kind of default parameter that explicitly wires the parameter to a property:

``` c#
  public abstract Person With(string firstName = this.FirstName, string lastName = this.LastName);
``` 

To explicitly facilitate interop with an existing type, a mandatory `With` method could be allowed to be provided as an extension method. It is unclear how that would work with the default parameter approach, though.

# Positional deconstruction
(see e.g. #206)

This feature allows a positional syntax for extracting the property values from an object, for instance in the context of pattern matching, but potentially also elsewhere.

Ideally, a positional deconstruction would simply generate an access of each member whose value is obtained:

``` c#
p is Person("Mickey", *)
```
becomes
``` c#
p.FirstName == "Mickey"
```

Again, this requires the compiler's understanding of how positions correspond to property names. Again, the same strategies as for object initializers are possible. See e.g. #8415.

Additionally, just as in with-expressions, one might wish to override the default behavior, or provide it if names don't match. Again, an explicit method could be used:

``` c#
public abstract class Person
{
  ...
  public void Person GetValues(out string firstName, out string lastName);
}
```
There are several options as to the shape of such a method. Instead of out-parameters, it might return a tuple. This has pros and cons: there could be only one tuple-returning `GetValues` method, because there would be no parameters to distinguish signatures. This may be a good or a bad thing.

Just as the `With` method, we can decide whether deconstruction should _require_ a `GetValues` method, or should fall back to metadata or to name matching against the constructor's parameter names.

If the `GetValues` method is used, the compiler doesn't need to resolve between positions and properties: the deconstruction as well as the method are already positional. We'd generate the code as follows:

``` c#
string __p1;
string __p2;
p.GetValues(out __p1, out __p2);
...
__p1 == "Mickey"
```

Somewhat less elegant for sure, and possibly less efficient, since the `LastName` is obtained for no reason. However, this is compiler generated code that no one has to look at, and it can probably be optimized, so this may not be a big issue.

# Summary

For each of these three features we are grappling with the position-to-property match. Our options:

1. Require specific metadata
2. Match property and parameter names, possibly in a case sensitive way
3. For deconstruction and with-expressions, allow or require specific methods (`GetValues` and `With` respectively) to implement their behavior, and possibly have special syntax in `With` methods to provide the name-to-position matching.

We continue to work on this.